using System;
using System.Globalization;
using System.IO;
using System.Text;
using SoftPi.Tariscope.Common;
using System.Data.Common;
using System.Data;

/// <summary>
/// Обработка расшифровки от оператора Lifecell (2021.09)
///  </summary>
public class Ua_Lifecell_2021 : IScript
{
    /// <summary>
    /// True для создание абонента если такового нет
    /// </summary>
    const bool CREATE_ABONENT = true;

    /// <summary>
    /// Кодировка файла (Cyrillic (Windows))
    /// </summary>
    readonly static Encoding ENCODING = Encoding.GetEncoding(1251);

    /// <summary>
    /// Язык файла
    /// </summary>
    readonly static CultureInfo CULTURE = CultureInfo.GetCultureInfoByIetfLanguageTag("ru-RU");

    /// <summary>
    /// Разделитель в csv файле
    /// </summary>
    const char CSV_SEPARATOR = ';';

    /// <summary>
    /// Имя услуги по предоставлению интернет услуг
    /// </summary>
    const string INTERNET_SERVICE_NAME = "Интернет услуги";


    //HEADERS
    const string NUMBER_A = "Номер A";
    const string SERVICE_LBL = "-";
    const string INTERNET_SERVICES = "internet";
    const string INCOMING_CALL = "входящие звонки";
    const string OUTGOING_CALL_1 = "исходящие звонки";
    const string OUTGOING_CALL_2 = "Переадресованные соединения";
    const string INCOMING_SMS = "SMS входящие";
    const string OUTGOING_SMS = "SMS исходящие";


    private IScriptDatabaseHelper DBHelper;
    private IScriptHost Host;

    private int CurrentAbonentId;

    public void Init(IScriptHost host, IScriptDatabaseHelper dbHelper)
    {
        this.Host = host;
        this.DBHelper = dbHelper;
    }

    public void Main(object fileName)
    {
        string[] fileContent = File.ReadAllLines(Convert.ToString(fileName), ENCODING);
        int lineIndex = 0;

        Logger.Status("start");

        try
        {
            foreach (var line in fileContent)
            {
                lineIndex += 1;

                if (Host.IsCancelling)
                    return;

                if (string.IsNullOrEmpty(line))
                    continue;

                try
                {
                    string[] columns = CsvHelper.SplitCSV(line, CSV_SEPARATOR);

                    if (columns.Length < 10)
                        continue;

                    if (columns[0] == NUMBER_A)
                    {
                        CurrentAbonentId = -1;
                    }
                    else if (CurrentAbonentId <= 0 && columns[0] != NUMBER_A && !string.IsNullOrWhiteSpace(columns[0]))
                    {
                        int departmentId = GetRootDepartment();
                        CurrentAbonentId = GetAbonentByDn(columns[0], departmentId);
                    }

                    if (CurrentAbonentId > 0 && columns[2] == SERVICE_LBL)
                    {
                        SaveService(columns);
                    }
                    else if (CurrentAbonentId > 0 && columns[2] == INTERNET_SERVICES)
                    {
                        SaveService(columns);
                    }
                    else if (columns[4] == OUTGOING_CALL_1 || columns[4] == OUTGOING_CALL_2)
                    {
                        SaveOutgoingCall(columns);
                    }
                    else if (columns[4] == INCOMING_CALL)
                    {
                        SaveIncomingCall(columns);
                    }
                    else if (columns[4] == INCOMING_SMS)
                    {
                        SaveIncomingSms(columns);
                    }
                    else if (columns[4] == OUTGOING_SMS)
                    {
                        SaveOutgoingSms(columns);
                    }
                }
                catch { }
            }
        }
        catch (Exception ex)
        {
            throw new Exception(string.Format("Exception in line #{0}", lineIndex), ex);
        }
    }

    #region DB insertions

    void SaveIncomingCall(string[] columns)
    {
        Host.AddCall(new CallModel
        {
            RecordType = RecordTypeEnum.Imported,
            CallDuration = Convert.ToInt32(GetDuration(columns[6]).TotalSeconds),
            CallDateTime = DateTime.Parse(columns[1], CULTURE),
            Originator = "L000000",
            Terminator = columns[0].Trim(),
            CLID = columns[2].Trim(),
            Cost = decimal.Parse(columns[9].Replace(',', '.'), CultureInfo.InvariantCulture),
            CallType = 6,
            ProjectCode = columns[5]
        });
    }

    void SaveOutgoingCall(string[] columns)
    {
        Host.AddCall(new CallModel
        {
            RecordType = RecordTypeEnum.Imported,
            CallDuration = Convert.ToInt32(GetDuration(columns[6]).TotalSeconds),
            CallDateTime = DateTime.Parse(columns[1], CULTURE),
            Originator = columns[0].Trim(),
            Terminator = "L000000",
            DialNumber = columns[2].Trim(),
            Cost = decimal.Parse(columns[9].Replace(',', '.'), CultureInfo.InvariantCulture),
            CallType = 6,
            ProjectCode = columns[5]
        });
    }

    void SaveOutgoingSms(string[] columns)
    {
        Host.AddCall(new CallModel
        {
            RecordType = RecordTypeEnum.Imported,
            CallDateTime = DateTime.Parse(columns[1], CULTURE),
            Originator = columns[0].Trim(),
            Terminator = "L000000",
            DialNumber = columns[2].Trim(),
            CLID = "",
            Cost = decimal.Parse(columns[9].Replace(',', '.'), CultureInfo.InvariantCulture),
            CallType = 13
        });
    }

    void SaveIncomingSms(string[] columns)
    {
        Host.AddCall(new CallModel
        {
            RecordType = RecordTypeEnum.Imported,
            CallDateTime = DateTime.Parse(columns[1], CULTURE),
            Originator = "L000000",
            Terminator = columns[2].Trim(),
            CLID = columns[0].Trim(),
            Cost = decimal.Parse(columns[9].Replace(',', '.'), CultureInfo.InvariantCulture),
            CallType = 13
        });
    }

    void SaveService(string[] columns)
    {
        Host.AddService(new ServiceModel
        {
            ServiceId = GetServiceId(columns[2] == INTERNET_SERVICES ? INTERNET_SERVICE_NAME : columns[4]),
            AbonentId = CurrentAbonentId,
            ServiceDate = DateTime.Parse(columns[1], CULTURE),
            Cost = decimal.Parse(columns[9].Replace(',', '.'), CultureInfo.InvariantCulture)
        });
    }

    #endregion

    #region Service methods

    int GetRootDepartment()
    {
        int nodeId = 0;
        using (IDbConnection cn = DBHelper.GetConnection())
        {
            DbCommand cmd = DBHelper.GetCommand("ab_mc_pbxdetails", cn, true);
            cmd.Parameters["@id"].Value = Host.GetProperty("PBXID");
            using (IDataReader rs = cmd.ExecuteReader())
            {
                if (rs.Read())
                    nodeId = Convert.ToInt32(rs["NodeId"]);
            }
            cmd = DBHelper.GetCommand("ab_mc_departmentroot", cn, true);
            cmd.Parameters["@nodeid"].Value = nodeId;
            using (IDataReader rs = cmd.ExecuteReader())
            {
                if (rs.Read())
                    return Convert.ToInt32(rs["ID"]);
                else
                    return 0;
            }
        }
    }

    int GetAbonentByDn(string dn, int rootDepartment)
    {
        int abonentID = 0;

        try
        {
            using (IDbConnection cn = DBHelper.GetConnection())
            {
                DbCommand cmd = DBHelper.GetCommand("ab_mc_abonentsearch2", cn);
                DBHelper.SetParameter("@dn", dn, cmd);

                using (IDataReader rs = cmd.ExecuteReader())
                {
                    if (rs.Read())
                    {
                        Logger.Status("Abonent found: {0}", rs["ID"]);
                        return Convert.ToInt32(rs["ID"]);
                    }
                    else if (CREATE_ABONENT)
                    {
                        rs.Close();

                        Logger.Status("Try to create abonent with dn - {0}", dn);

                        cmd = DBHelper.GetCommand("ab_mc_abonentadd", cn);
                        DBHelper.SetParameter("@lastname", string.Format("a.{0}", dn), cmd);
                        DBHelper.SetParameter("@departmentid", rootDepartment, cmd);
                        cmd.ExecuteNonQuery();
                        abonentID = Convert.ToInt32(DBHelper.GetParameter(0, cmd));

                        cmd = DBHelper.GetCommand("ab_mc_dnadd", cn);
                        cmd.Parameters["@abonentid"].Value = abonentID;
                        cmd.Parameters["@dn"].Value = dn;
                        cmd.ExecuteNonQuery();
                    }
                }
            }
        }
        catch (Exception e)
        {
            Logger.Exception(e);
        }

        return abonentID;
    }

    int GetServiceId(string serviceName)
    {
        using (IDbConnection cn = DBHelper.GetConnection())
        {
            DbCommand cmd = DBHelper.GetQuery("SELECT ID FROM Services WHERE Name=@name", cn);
            DBHelper.AddParameter("@name", serviceName, cmd);
            using (IDataReader rs = cmd.ExecuteReader())
            {
                if (rs.Read())
                    return rs.GetInt32(0);
            }
            // no service found, create one
            cmd = DBHelper.GetCommand("ab_mc_serviceadd", cn, true);
            cmd.Parameters["@name"].Value = serviceName;
            cmd.Parameters["@period"].Value = 0;
            cmd.ExecuteNonQuery();
            return System.Convert.ToInt32(cmd.Parameters[0].Value);
        }
    }

    TimeSpan GetDuration(string durationString)
    {
        try
        {
            durationString = durationString.Trim();
            TimeSpan duration = new TimeSpan();
            int delimitersCount = GetOccurencesCount(":", durationString);
            switch (delimitersCount)
            {
                case 2:
                    {
                        string[] parts = durationString.Split(':');
                        return new TimeSpan(System.Convert.ToInt32(parts[0]), System.Convert.ToInt32(parts[1]), System.Convert.ToInt32(parts[2]));
                    }
                case 1:
                    {
                        string[] parts = durationString.Split(':');
                        return new TimeSpan(0, System.Convert.ToInt32(parts[0]), System.Convert.ToInt32(parts[1]));
                    }
                case 0:
                    {
                        return new TimeSpan(0, 0, System.Convert.ToInt32(durationString));
                    }
                default:
                    {
                        throw new Exception();
                        break;
                    }
            }
        }
        catch { }

        return TimeSpan.FromSeconds(0);
    }

    int GetOccurencesCount(string needle, string haystack)
    {
        return Convert.ToInt32((haystack.Length - haystack.Replace(needle, string.Empty).Length) / (double)needle.Length);
    }
    #endregion
}
